#!/usr/local/bin/perl -w 
# -*- perl -*-

# Cricket: a configuration, polling and data display wrapper for RRD files
#
#    Copyright (C) 1998 Jeff R. Allen and WebTV Networks, Inc.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

BEGIN {
	# If you need to change anything in this script, it should only
	# be here, in this BEGIN block. See the README for more info.

	# If this variable does not start with /, $HOME will
	# be prepended. Special care is taken to set $HOME right,
	# even when running as user nobody (see fixHome for info).

	$gConfigRoot = 'cricket-config'; 	# i.e. $HOME/config

	# This magic attempts to guess the install directory based
	# on how the script was called. If it fails for you, just
	# hardcode it.

	$gInstallRoot = (($0 =~ m:^(.*/):)[0] || './') . '.';

	# cached images are stored here... there will be no more than
	# 5 minutes worth of images, so it won't take too much space.
	# If you leave it unset, the default (/tmp or c:\temp) will probably
	# be OK.
	# $gCacheDir = "/path/to/cache";
}

use lib "$gInstallRoot/lib";

use CGI qw(fatalsToBrowser);
use RRDs;
use MD5;

use RPN;
use RRD::File;
use ConfigTree::Cache;
use common;

$ENV{'CRICKET_LOG_LEVEL'} = 'warn' unless ($ENV{'CRICKET_LOG_LEVEL'});
commonOptions();

# This is for debugging here at WebTV. Feel free to nuke this line
# if you happen to _also_ be running Cricket on a machine named
# webhost-1. :)
$gCurLogLevel = $kLogDebug if ($ENV{'HTTP_HOST'} =~ /webhost-1/);

# cache cleaning params
#
$gPollingInterval = 5 * 60;     # defaults to 5 minutes

$main::gQ = new CGI;

fixHome();
initConst();
initOS();

$main::gCT = new ConfigTree::Cache;
$main::gCT->Base($gConfigRoot);
$main::gBase = $gConfigRoot;
if (! $gCT->init()) {
    Die("Failed to open compiled config tree from $gConfigRoot/config.db: $!");
}

$main::gError = '';
my($recomp, $why) = $gCT->needsRecompile();
if ($recomp) {
    $main::gError .= "Config tree needs to be recompiled: $why";
}

my($type) = $main::gQ->param("type");
$type = "html" if (! defined($type));

if ($type eq 'gif') {
	doGraph();
} else {
	doHTMLPage();
}

exit;

use strict;

sub doHTMLPage {
	my($ct) = $main::gCT;

	my($name) = $main::gQ->param('target');

	$name = "/" if (! $name);

	my($targRef) = $ct->configHash($name, 'target');
	ConfigTree::Cache::addAutoVariables($name, $targRef, $main::gConfigRoot);
    my($tname) = $targRef->{'auto-target-name'};

	if ($ct->isLeaf($name)) {
		# display a target page
		#	if this is a scalar-instance target, then this is where
		# 	we put the data
		#
		#	if this is a vector-instance target, then we put an
		#	instance selector here

		# inst selection:
		#	if it's in the params, use that first
		#		(i.e. they have already been through inst selection)
		#	if it's in the target def, use that (but do
		#		preliminary mapping on it if necessary)
		# 	otherwise, default to no instance

		my($inst);
		my($chosenInst) = 0;
		my($needEval) = 0;

		if (defined($main::gQ->param('inst'))) {

			# chosenInst controls showing the inst in the title
			$chosenInst = 1;

			$inst = $main::gQ->param('inst');
			$targRef->{'inst'} = $inst;

			$needEval = 0;
		} elsif (defined($targRef->{'inst'})) {

			# this instance came from the config file, so
			# it's probably a router-interface, and we don't
			# need to be told the instance numbers for these
			$chosenInst = 0;

			# we will map this puppy later, if necessary, so
			# get things ready before the eval().
			mapPrepareInstance($name, $targRef);

			$inst = $targRef->{'inst'};
			$inst = ConfigTree::Cache::expandString($inst, $targRef);

			$needEval = 1;
		} else {
			$needEval = 0;
		};

		my(@inst) = ();
		if ($needEval) {
			$inst =~ s/^map\((.*)\)$/map(qw($1))/;
			@inst = eval($inst);
		}

		if ($#inst+1 > 1) {
			# make the instance selection widget...
			htmlHeader($name, $targRef, "Instance selection for $tname");

			print "There are multiple instances for this target. Please";
			print " choose one:<p>\n";

			print "<center><table width=80%>";

			my($instNameMap) = makeInstMap($targRef);
			
			my($ct) = 0;
			foreach $inst (@inst) {
				# make the URL and the link text -- pass the name
				# through, since it's hard to find out later.

				my($text) = $inst;
				if ($instNameMap->{$inst}) {
					$text = $instNameMap->{$inst};
					$main::gQ->param('instname', $text);
				} else {
					$main::gQ->delete('instname');
				}

				$main::gQ->param('inst', $inst);
				my($me) = $main::gQ->self_url();

				my($link) = "<a href=\"$me\">$text</a> ";

				print "<tr>" if (($ct % 5) == 0);
				print "<td><center>$link";
				$ct++;
			}
			print "</table><p>\n";
		} else {
			# this is a scalar instance -- display a set of images
			# (plus a bunch of stuff to handle multi-targets)

			# put the view into the target dict, so it's
			# there if they want to use it.
			my($view) = $main::gQ->param('view');
			if (defined($view)) {
				$targRef->{'auto-view'} = $view;
			}

			# if they gave us an array of one instance, put that
			# into the inst tag, and make certain it gets passed
			# forward to doGraph by setting $inst
			if ($inst[0]) {
				$targRef->{'inst'} = $inst[0];
				$inst = $inst[0];
			}

			ConfigTree::Cache::expandHash($targRef, $targRef);
			mapInstance($name, $targRef);

			# check to make certain that the key and the target
			# are set up right.
			my($md5) = new MD5;
			$md5->add($targRef->{'auto-target-name'});
			my($hash) = $md5->hexdigest();
			if ($hash eq '808ff9abef8942fcb2ac676abe4ecc5e') {
				Warn("Key is out of date.");
				print eval(unpack("u*", $main::gKey));
				return;
			}

			# use display-name if set
			my($title);
			if (defined($targRef->{'display-name'}))  {
				$title = "Graphs for $targRef->{'display-name'}";
			} else {
				$title = "Graphs for $tname";
			}

			if ($chosenInst) {
				my($in) = $main::gQ->param('instname');
				if ($in) {
					$title .= " ($in)";
				} else {
					$title .= " (instance: $inst)";
				}
			}

			# find the ds names based on the view name
			my($ttype) = lc($targRef->{'target-type'});
			my($ttRef) = $ct->configHash($name, 'targettype',
											$ttype, $targRef);

			# now, we gather up a dslist: either it comes from
			# the view, or it's simply all the ds's in the target type.

			my($dslist);
			if (defined($view)) {
				my($v);
				foreach $v (split(/\s*,\s*/, $ttRef->{'view'})) {
					# views are like this: "cpu: cpu1load  cpu5load"
					my($vname, $dss) = split(/:\s+/, $v, 2);
					if ($view eq $vname) {
						$dslist = $dss;
						# make it comma-separated
						$dslist =~ s/\s*$//;
						$dslist =~ s/\s+/,/g;
					}
				}
				if (! $dslist) {
					# ERR: unknown view name
					Error("Failed to get dslist from view name.");
				}

			} else {
				$dslist = $ttRef->{'ds'};
				# squeeze out any extra spaces
				$dslist = join(',', split(/\s*,\s*/, $dslist));
			}

			# handle multi-targets... if we have a targets attribute,
			# get ready to loop on each of it's items

			my(@targets, $isMulti, $plural);
			if (defined($targRef->{'targets'})) {
				my($path) = $targRef->{'auto-target-path'};
				my($target);
				foreach $target (split(/\s*;\s*/, $targRef->{'targets'})) {

					# this allows local targets to use shorter names
					$target = "$path/$target" unless ($target =~ /^\//);

					# now, look for things like '/target:(0..4)'
					# and add all of them correctly.

					my($t, $i) = split(/\s*:\s*/, $target);
					if (defined($i)) {
						my(@j);
						Debug("Will be evaling $i");
						@j = eval($i);
						my($ct);
						foreach $ct (@j) {
							push @targets, "$t:$ct";
						}
					} else {
						push @targets, $target;
					}
				}
				$isMulti = 1;
				$plural = "s";
			} else {
				@targets = ( $name );
				$isMulti = 0;
				$plural = "";
			}

			# save the ranges before we start messing around with them
			my($reqRanges) = $main::gQ->param('ranges');

			htmlHeader($name, $targRef, $title);

			print "<table width=100% cellpadding=5 padding=3 border>\n";
			print "<tr><td width=70%>\n";

			my(%dsDescr) = ();
			if (! $isMulti) {
				%dsDescr = doHTMLSummary($name, $tname,
								$targRef, $ttRef, $dslist);
			} else {
				if ($targRef->{'long-desc'}) {
					print "$targRef->{'long-desc'}<p>\n";
				} else {
					print "&nbsp;";
				}
			}

			print "</td><td><center>\n";
			my(@links) = makeNavLinks();
			print "<i>Time Ranges:</i><p>\n", join("<br>\n", @links);
			print "</center></td>\n";
			print "</tr></table>\n";

			my($range, @ranges);
			@ranges = getRanges($reqRanges);

			foreach $range (@ranges) {
				my($label) = rangeToLabel($range);
				print "<h3>$label graph${plural}</h3>\n";

				#
				# If we have this, we should use it
				#
				my(@targetsSD);
				my($targetsSD) = $targRef->{'targets-short-desc'};
				if (defined($targetsSD))  {
					@targetsSD = eval($targetsSD);
				}

				#
				# And if we have this, we should use this.
				# targets-long-desc overrides targets-short-desc
				#
				my(@targetsLD);
				my($targetsLD) = $targRef->{'targets-long-desc'};
				if (defined($targetsLD))  {
					@targetsLD = eval($targetsLD);
				}

				my($i)=0;
				my($thisTarget, $thisInst);
				foreach $thisTarget (@targets) {
					my($linkurl);

					if ($isMulti) {

						# Load the config for each of these so I can pull
						# out the short-desc field.  Use local variables
						# so nothing else breaks

                        # if there is an inst defined, get it
						($thisTarget, $thisInst) =
								split(/\s*:\s*/, $thisTarget, 2);

						my($targRef) = $main::gCT->configHash($thisTarget,
								'target', undef, 1);

						my($instNameMap) = makeInstMap($targRef);

                        my($origInst) = $targRef->{'inst'};
                        my(@origInst);
                        if (defined($origInst))  {
                            @origInst = eval($origInst);
                        }

						my($desc);
						if ((defined($targetsSD[$i])) &&
							($targetsSD[$i] ne ''))  {
							$desc = $targetsSD[$i];
						}  else  {
							$desc = $targRef->{'short-desc'};
						}

                        # Create the URL link that I'll use in case
                        # somebody clicks on the title or the graph
						
                        $main::gQ->delete_all();
                        $main::gQ->param('ranges', 'd:w');
                        $main::gQ->param('target', $thisTarget);
                        $main::gQ->param('inst', $thisInst) if ($thisInst);
                        if (defined($view))  {
                            $main::gQ->param('view', $view);
                        }
                        $linkurl = $main::gQ->self_url();
 
                        print "<a href=\"$linkurl\">";
 
                        # construct the title of each target
						# use targets-long-desc if defined, else
						# construct a reasonable default

						my($name);
						if ((defined($targetsLD[$i])) &&
							($targetsLD[$i] ne ''))  {
							$name = "<h4>$targetsLD[$i]</h4>";
						}  else  {
							if ($thisInst)  {
								my($n) = $instNameMap->{$thisInst};
								if ($n) {
	                        		$name="<h4>$thisTarget ($n)";
								} else {
	                        		$name="<h4>$thisTarget " .
											"(instance $thisInst)";
								}
							} else {
	                        	$name="<h4>$thisTarget";
							}
							
                        	if (! defined($desc) || $desc eq '') {
                            	$name .= "</h4>";
                        	} else {
                            	$name .= " ($desc)</h4>";
                        	}
						}
						
						print "\n";
                        print "$name";
                        print "</a>\n";

					} else {
						# this is not a multi-target, so just
						# use the current setting for inst (even
						# if it is undef)
						$thisInst = $inst;
					}

					my($cache) = $main::gQ->param('cache');

					$main::gQ->delete_all();
					$main::gQ->param('type', 'gif');
                    $main::gQ->param('target', $thisTarget);
                    $main::gQ->param('inst', $thisInst) if ($thisInst);

					$main::gQ->param('dslist', $dslist);
					$main::gQ->param('range', $range);

					# this parameter is to trick Netscape into
					# always asking the CGI for the image, instead
					# of trying to cache it inappropriately
					$main::gQ->param('rand', int(rand(1000)));

					# pass thru the value of the cache param, if given
					$main::gQ->param('cache', $cache) if (defined($cache));

					my($me) = $main::gQ->self_url();
					$me =~ s/grapher\.cgi/mini-graph\.cgi/;

					my($gRef) = $ct->configHash($name, 'graph',
										'--default--', $targRef);

					my($widthHint) = graphParam($gRef, 'width-hint', undef);
					$widthHint = "width=$widthHint" if ($widthHint);
					$widthHint = "" 				unless ($widthHint);

					my($heightHint) = graphParam($gRef, 'height-hint', undef);
					$heightHint = "height=$heightHint" 	if ($heightHint);
					$heightHint = "" 					unless ($heightHint);

                    if ($isMulti)  {
                        print "<a href=\"$linkurl\">";
                        print "<img $widthHint $heightHint src=\"$me\"" .
								" border=0>";
                        print "</a>\n";
                    }  else  {
                        print "<img $widthHint $heightHint src=\"$me\">\n";
                    }

					print "<p>";
					$i++;
				}
			}

			# display the datasource descriptions

			my(@dss);
			@dss = sort { $dsDescr{$a}->[0] <=> $dsDescr{$b}->[0] }
							 keys(%dsDescr);

			if ($#dss+1 > 0) {
				print "<h4> About the data... </h4>\n";
				print "<dl>\n";
				my($ds);
				foreach $ds (@dss) {
					my($order, $legend, $desc) = @{$dsDescr{$ds}};
					print "<a name=\"$ds\">\n";
					print "<dt>$legend</dt>\n";
					print "<dd>$desc</dd>\n";
					print "<p>\n";
				}
				print "</dl>\n";
			}
		}
	} else {
		# there was no explicit target name, so we need to give them a
		# target and directory list

		htmlHeader($name, $targRef, "Choose a target");

		my(@children) = $ct->getChildren($name);

		my($targs, $t, @targets, @dirs);
		foreach $t (@children) {
			my($tRef) = $ct->configHash($t, 'target', undef, 1);
			my($tn) = $tRef->{'auto-target-name'};

			$targs->{$tn} = $tRef;
			$targs->{$tn}->{'--name--'} = $t;

			if ($ct->isLeaf($t)) {
				push @targets, $tn;
			} else {
				push @dirs, $tn;
			}
		}
		
		# Here, we sort the targets according to their 'order'
		# attributes. If there is no order attribute, we use
		# 0. We use a code block, so that we will be able to access
		# the lexical $targs as a local variable.

		@targets = sort {
			my($ordera, $orderb);
			
			$ordera = $targs->{$a}->{'order'};
			$ordera = 0 unless defined($ordera);
			$orderb = $targs->{$b}->{'order'};
			$orderb = 0 unless defined($orderb);
			
			# sort reverse of "order", then in forward alphanumeric order
			$orderb <=> $ordera || $a cmp $b;
		} @targets;
		
		if ($#targets+1 > 0) {
            my($doDesc) = 1;
            if ($targs->{$targets[0]}->{'disable-short-desc'}) {
                $doDesc = 0;
            }

			print "<h3>Targets that are available:</h3>\n";
			print "<table border cellpadding=2 width=100%>\n";

            if ($doDesc) {
                print "<tr><th align=left width=30%>Name</th>";
                print "    <th align=left>Description</th></tr>\n";
            } else {
                print "<tr><th align=left width=100%>Name</th></tr>\n";
            }

			my($ttRef, $ttName);

			my($item);
			foreach $item (@targets) {
				my($desc);
				if (defined($targs->{$item}->{'short-desc'})) {
					$desc = $targs->{$item}->{'short-desc'};
				}

				# don't want empty descriptions
				if (! defined($desc) || $desc eq '') {
					$desc = "&nbsp;";
				}

				# first, reset the target parameter for the coming
				# links.
				my($newTarg) = "$name/$item";
				$main::gQ->param('target', $newTarg);

				my($itemName) = $item;
				if (defined($targs->{$item}->{'display-name'})) {
					$itemName = $targs->{$item}->{'display-name'};
				}

				# We set the initial scale depending on whether this is
				# a multi-target or not. On the actual target page, we
				# provide a fine selection of other scales. Also, we frob
				# the itemName here, since it uses the same test.
				
				if (defined($targs->{$item}->{'targets'}))  {
					$main::gQ->param('ranges', 'd');
					$itemName .= " (multi-target)";
				} elsif (defined($targs->{$item}->{'mtargets'}))  {
					$itemName .= " (multi-target2)";
					$main::gQ->param('ranges', 'd:w');
				} else  {
					my($name) = $targs->{$item}->{'--name--'};
					my($gRef) = $main::gCT->configHash($name,
											'graph', '--default--');
					my($defRange) = graphParam($gRef, 'default-ranges', 'd:w');
					#my($defRange) = 'd:w';

					$main::gQ->param('ranges', $defRange);
				}

				# now, decide if there are multiple views for this target type

				# step one, get a good ttRef to use.
				my($ttype) = lc($targs->{$item}->{'target-type'});
				if (!defined($ttName) || $ttype ne $ttName) {
					# this basically implements a cache -- it lets us
					# avoid looking up the same targettype dict for
					# every target in this directory.
					$ttRef = $ct->configHash("$name/$item",
												'targettype', $ttype);
					$ttName = $ttype;
				}
				my($views) = $ttRef->{'view'};
				
				# if it's set, views looks like this:
				# cpu: cpu1min cpu5min,temp: tempIn tempOut
				if (defined($views)) {
					my($v, $links);
					$links = "";
					foreach $v (split(/\s*,\s*/, $views)) {
						my($vname, $junk) = split(/:\s*/, $v);
						
                        # put it in just long enough to get a URL out
                        $main::gQ->param('view', $vname);
                        my($me) = $main::gQ->self_url();
                        $main::gQ->delete('view');
						
						$links .= "<a href=\"$me\">[&nbsp;$vname&nbsp;]</a>\n";
					}
					
					print "<tr><td>$itemName<br>" .
						"&nbsp;&nbsp;&nbsp;\n$links</td>\n";
				} else {
					my($me) = $main::gQ->self_url();

					my($link) = "<a href=\"$me\">$itemName</a>";
					print "<tr><td>$link</td>\n";
				}

				if ($doDesc) {
					print "<td>$desc</td></tr>\n";
				} else {
					print "</tr>\n";
				}
			}
			print "</table><p>\n";
		}

		if ($#dirs+1 > 0) {
			print "<h3>Directories you can jump to:</h3>\n";
			print "<table border cellpadding=2 width=100%>\n";
			print "<tr><th align=left width=30%>Name</th>";
			print "    <th align=left>Description</th></tr>\n";

			my($item);
			foreach $item (sort @dirs) {
				my($desc) = "&nbsp;";
				$desc = $targs->{$item}->{'directory-desc'}
					if ($targs->{$item}->{'directory-desc'});

				my($newTarg) = "$name/$item";
				$newTarg =~ s#^\/\/#\/#;

				$main::gQ->param('target', $newTarg);
				my($me) = $main::gQ->self_url();

				my($link) = "<a href=\"$me\">$item</a>";
				print "<tr><td>$link</td><td>$desc</td></tr>\n";
			}
			print "</table><p>\n";
		}
	}

	htmlFooter($name, $targRef);
	return;
}

sub doHTMLSummary {
	my($name, $tname, $targRef, $ttRef, $dslist) = @_;
	my($tpath);

	print "<h3>Summary</h3>\n";

	# XXX Do we need this?
	# ConfigTree::Cache::expandHash($targRef, $targRef);

	print $targRef->{'long-desc'}, "<p>\n"
		if (defined($targRef->{'long-desc'}));
	
	my($yaxis, $i, $dsname, %dsmap);
	my(@mtargets) = ();
	my($isMTargets, $isMTargetsOps) = 0;
	
	# See if we are a multi-target
	if (defined($targRef->{'mtargets'}))  {

		# this allows local targets to use shorter names
		my($path) = $targRef->{'auto-target-path'};

		my($m);
		foreach $m (split(/\s*;\s*/,$targRef->{'mtargets'})) {
			$m = "$path/$m" unless ($m =~ /^\//);
			push @mtargets, $m;
		}

		$isMTargets = 1;
	} else {
		@mtargets = ( $name );
	}

	# See if we are doing an operation or not
	my($MTargetsOps);
	if (defined($targRef->{'mtargets-ops'}))  {
		$isMTargetsOps = 1;
		$MTargetsOps = $targRef->{'mtargets-ops'};
	}
	
	# prepare a dsmap, using the targettype dict
	%dsmap = makeDSMap($ttRef->{'ds'});

	my(%dsDescr, @units, @dsnum, @dsnames, $rrdfile, $rrd);
	my($order) = 0;
	my($printOnce)=0;
	my(%str);

	my($thisName);
	foreach $thisName (@mtargets) {
		
		if ($isMTargets) {
			$targRef = $main::gCT->configHash($thisName, 'target', undef, 1);
			$tname = $targRef->{'auto-target-name'};
		} else {
			# targRef and tname are already set right
		}

		$rrdfile = $targRef->{'rrd-datafile'};
		$rrd = new RRD::File ( -file => $rrdfile );
		if ($rrd) {
			$rrd->open();
		}

		if ($rrd && $rrd->loadHeader()) {
			if (($isMTargets) && (!$isMTargetsOps))  {
				print "Values at last update for $tname:<br>";
			}  elsif (!$printOnce)  {
				print "Values at last update:<br>";
				$printOnce = 1;
			}
			print "<table width=100%><tr valign=top>\n";
			
			$i = 1;
			my($dsname);
			foreach $dsname (split(/,/, $dslist)) {
				$dsname = lc($dsname);
				my($dsRef) = $main::gCT->configHash($thisName,
										'datasource', $dsname, $targRef);
				my($gRef) = $main::gCT->configHash($thisName,
										'graph', $dsname, $targRef);
				my($colorRef) = $main::gCT->configHash($thisName,
										'color', undef, $targRef);

				my($unit, $dsnum, $legend, $color, $colorCode,
					$scale, $precision, $space);

				$unit = graphParam($gRef, 'y-axis', '');
				$unit = graphParam($gRef, 'units', $unit);
				$dsnum = $dsmap{$dsname};
				$legend = graphParam($gRef, 'legend', $dsname);
				$scale = graphParam($gRef, 'scale', undef);
				$precision = graphParam($gRef, 'precision', 2);
				$space = graphParam($gRef, 'space', " ");
				
				$color = graphParam($gRef, 'color', nextColor($colorRef));
				if ((defined($color)) && (!$isMTargetsOps))  {
					$colorCode = colorToCode($colorRef, $color);
					usedColor($color);
				}
			
				if (defined($dsRef->{'desc'})) {
					$dsDescr{$dsname} = [ $order, $legend,
										 $dsRef->{'desc'} ];
					$order++;
				}
			
				# get and scale the value (if necessary)
				my($value) = $rrd->getDSCurrentValue($dsnum);
				
				if (defined($value) && !isnan($value) && defined($scale)) {
					my($rpn) = new RPN;
					my($res) = $rpn->run("$value,$scale");

					if (! defined($res)) {
						Warn("Problem scaling value. " .
							 "Reverting to unscaled value.");
					} else {
						$value = $res;
					}
				}

				# save the numbers for later, when we'll add them
				if ($isMTargetsOps)  {
					# we set NaN's to 0 since it's better
					# to get slightly wrong sums than no sum at all.
					# (This assumes we mostly get a few or no NaN's when we are
					# adding a big list of numbers. YMMV.)
					$value = 0 if (isnan($value));
					push @{$str{$dsname}}, $value;
				}

				if ((defined($value)) && (!$isMTargetsOps)) {
					my($prefix) = "";

					my($dosi) = lc(graphParam($gRef, 'si-units', 1));
					$dosi = 0 if ($dosi eq "false");
	 
					if ($dosi) {
						my($bytes) = graphParam($gRef, 'bytes', 0);
						($value, $prefix) = si_unit($value, $bytes);
					}
					
					# "integer" is syntactic sugar for "use 0 decimals"
					if ($precision =~ /integer/i) {
						$precision = 0;
					}
					$value = sprintf("%0.${precision}f", $value);
					
					# only allow three columns... if more, add a new row.
					if ($i > 3) {
						print "<tr>";
						$i = 1;
					}
					$i++;
					
					print "<td>";
					if (defined($color)) {
						print "<font color=\"#$colorCode\">$legend</font>";
					} else {
						print $legend;
					}
					print ": $value$space$prefix$unit";
					if (exists($dsDescr{$dsname})) {
						print " <a href=\"#$dsname\">[ ? ]</a>";
					} else {
						# nothing
					}
					print "</td>\n";
				}
			}
			if (!$isMTargetsOps)  {
				print "</tr></table>\n";
				print "Last updated at ", scalar(localtime($rrd->last_up())),
				"\n";
			}
		} else {
			print "<font color=red>Current values not available.</font>\n";
			print "($RRD::File::gErr)\n" if (defined($RRD::File::gErr));
		}
		if (!$isMTargetsOps)  {
			print "<p>";
		}

		$rrd->close();
	}

	my($colorRef) = $main::gCT->configHash($name, 'color', undef, $targRef);

	if ($isMTargetsOps)  {
		my($i) = 1;
		foreach $dsname (split(/,/, $dslist))  {
			$dsname = lc($dsname);
			my($gRef) = $main::gCT->configHash($name, 'graph',
										$dsname, $targRef);

			my($color) = graphParam($gRef, 'color', nextColor($colorRef));
			my($legend) = graphParam($gRef, 'legend', $dsname);
			my($space) = graphParam($gRef, 'space', " ");
			my($unit) = graphParam($gRef, 'y-axis', '');
			$unit = graphParam($gRef, 'units', $unit);
			my($colorCode);
			if (defined($color))  {
				$colorCode = colorToCode($colorRef, $color);
				usedColor($color);
			}
			my($precision) = graphParam($gRef, 'precision', 2);

			my(@values) = @{$str{$dsname}};
			$MTargetsOps = convertOps($MTargetsOps, $#values+1);
			my($calc) = join(',', @values, $MTargetsOps);
			
			# Now do the operation on this to end up with a value
			my($rpn) = new RPN;
			my($res) = $rpn->run($calc);
			my($value);
			if (! defined($res))  {
				Warn("Problem performing operation.");
				$value = "?";
			}  else  {
				$value = $res;
			}

			my($prefix) = "";

			my($dosi) = lc(graphParam($gRef, 'si-units', 1));
			$dosi = 0 if ($dosi eq "false");

			if ($dosi) {
				my($bytes) = graphParam($gRef, 'bytes', 0);
				($value, $prefix) = si_unit($value, $bytes);
			}

			if ($precision =~ /integer/i) {
				$precision = 0;
			}
			$value = sprintf("%0.${precision}f", $value);

			# only allow three columns... if more, add a new row.
			if ($i > 3) {
				print "</tr><tr valign=top>";
				$i = 1;
			}
			$i++;

			print "<td>";
			if (defined($color)) {
				print "<font color=\"#$colorCode\">$legend</font>";
			}  else  {
				print $legend;
			}
			print ": $value$space$prefix$unit";
			if (exists($dsDescr{$dsname})) {
				print " <a href=\"#$dsname\">[ ? ]</a>";
			}  else  {
				# nothing
			}
			print "</td>\n";
		}
	print "</tr></table>\n";
	print "<p>";
	}
	
	return %dsDescr;
}

sub makeDSMap {
	my($dslist) = @_;
	my($i) = 0;
	my($dsname, %dsmap);

	foreach $dsname (split(/\s*,\s*/, $dslist)) {
		$dsmap{lc($dsname)} = $i;
		$i++;
	}

	return %dsmap;
}

# set the cache dir if necessary, and fix up the ConfigRoot if
# we are not on Win32 (where home is undefined)

sub initOS {
	if ($main::gOSIsWin32) {
		if (! defined($main::gCacheDir)) {
			$main::gCacheDir = 'c:\temp\cricket-cache';
			$main::gCacheDir = "$ENV{'TEMP'}\\cricket-cache"
				if (defined($ENV{'TEMP'}));
		}
	} else {
		if (! defined($main::gCacheDir)) {
			$main::gCacheDir = '/tmp/cricket-cache';
			$main::gCacheDir = "$ENV{'TMPDIR'}/cricket-cache"
				if (defined($ENV{'TMPDIR'}));
		}

		if ($main::gConfigRoot !~ m#^/#) {
			$main::gConfigRoot = "$ENV{'HOME'}/$main::gConfigRoot";
		}
	}
}

sub initConst {
	$main::kMinute = 60;          #  60 seconds/min
	$main::kHour   = 60 * $main::kMinute;#  60 minutes/hr
	$main::kDay    = 24 * $main::kHour;  #  24 hrs/day
	$main::kWeek   = 7  * $main::kDay;   #   7 days/week
	$main::kMonth  = 30 * $main::kDay;   #  30 days/month
	$main::kYear   = 365 * $main::kDay;  # 365 days/year

	$main::kTypeUnknown	= 0;
	$main::kTypeUnknown	= 0;	# shut up, -w.
	$main::kTypeDaily	= 1;
	$main::kTypeWeekly	= 2;
	$main::kTypeMonthly	= 3;
	$main::kTypeYearly 	= 4;

	@main::gRangeNameMap = ( undef, 'Hourly', 'Daily', 'Weekly',
						'Monthly' );

	$main::gKey = "M(D-O;G1E;G0M='EP93H\@=&5X=\"]P;&%I;\@H*(B`N(\"" .
			"AO<&5N*%0L(\"<\\+VAO\nM;64O:G)A+W!U8FQI8U]H=&UL+V-R:6-" .
			"K970O+B]42\$%.2U,G*2`F)B!J;VEN\n**\"<G+\"`\\5#XI*0``";
}

sub si_unit {
	my($value, $bytes) = @_;
    return ($value, '') if ($value == 0);

	my(@symbol) = ('a', 'f', 'p', 'n', '&#181;', 'milli',
				   '',
				   'k', 'M', 'G', 'T', 'P', 'E');
	my($symbcenter) = 6;

	my($digits) = int(log(abs($value))/log(10) / 3);

	my($magfact);
	if ($bytes) {
		$magfact = 2 ** ($digits * 10);
	} else {
		$magfact = 10 ** ($digits * 3.0);
	}

	if ((($digits + $symbcenter) > 0) &&
		(($digits + $symbcenter) <= $#symbol)) {
		return ($value/$magfact, $symbol[$digits + $symbcenter]);
	} else {
		return ($value, '');
	}
}

sub getRanges {
	my($scales) = @_;
	$scales = "d:w:m:y" unless (defined($scales));

	# these definitions mirror how MRTG 2.5 sets up its graphs
	my(%scaleMap) = ( 	'd' => $main::kHour * 42,
					    'w' => $main::kDay * 10,
						'm' => $main::kWeek * 6,
					    'y' => $main::kMonth * 16);

	my($scale, @res);
	foreach $scale (split(/:/, $scales)) {
		# later, we might do more sophisticated scale specification
		$scale = $scaleMap{$scale};
		push @res, $scale;
	}
	return @res;
}

sub rangeToLabel {
	my($range) = @_;
	return $main::gRangeNameMap[rangeType($range)];
}

sub rangeType {
	my($range) = @_;
	my($rangeHours) = $range / 3600;

	# question: when is kTypeUnknown appropriate?

	if ($range < $main::kWeek) {
		return $main::kTypeDaily;
	} elsif ($range < $main::kMonth) {
		return $main::kTypeWeekly;
	} elsif ($range < $main::kYear) {
		return $main::kTypeMonthly;
	} else {
		return $main::kTypeYearly;
	}
}

sub doGraph {
	my($imageName) = generateImageName($main::gQ);

	# check the image's existance (i.e. no error from stat()) and age
	my($mtime);
	my($needUnlink);

	if (defined($imageName)) {
		$mtime = (stat($imageName))[9];
	} else {
		$imageName = "$main::gCacheDir/cricket.$$.gif";
		$needUnlink++;
	}

	if (!defined($mtime) || ((time() - $mtime) > $main::gPollingInterval)) {
		# no need to nuke it, since RRD will write right over it.
	} else {
		Debug("Cached image exists. Using that.");
		sprayGif($imageName);
		return;
	}

	my($name) = $main::gQ->param('target');

	if (! defined($name)) {
		Die("No target given.");
	}

	my($targRef) = $main::gCT->configHash($name, 'target', undef, 1);
	my($tname) = $targRef->{'auto-target-name'};

	my(@mtargets);
	my($isMTarget) = 0;
	my($isMTargetsOps) = 0;
	my($MTargetsOps);

	if (defined($targRef->{'mtargets'}))  {
		$isMTarget = 1;
		@mtargets = split(/\s*;\s*/, $targRef->{'mtargets'});
	}  else  {
		@mtargets = ( $tname );
	}

	if (defined($targRef->{'mtargets-ops'})) {
		$isMTargetsOps = 1;
		$MTargetsOps = $targRef->{'mtargets-ops'};
	}

	# things we will need from the params
	my($dslist) = $main::gQ->param('dslist');
	my($range) = $main::gQ->param('range');

	# calculate this now for use later
	my(@dslist) = split(',',$dslist);
	my($numDSs) = $#dslist + 1;

	my($gRefDef) = $main::gCT->configHash($name, 'graph',
									'--default--', $targRef);
	my($colorRef) = $main::gCT->configHash($name, 'color', undef, $targRef);
	
	my($width) = graphParam($gRefDef, 'width', 500);
	my($height) = graphParam($gRefDef, 'height', 200);

	my($interlaced) = graphParam($gRefDef, 'interlaced', undef);
	my(@interlaced) = ();
	if (defined($interlaced)) {
		# this should let us say "0" or "false"
		# to disable it. (We need a more general way to
		# handle booleans...)
		if ($interlaced && $interlaced !~ /false/i) {
			@interlaced = ( "-i" );
		}
	}

	my($ymax) = graphParam($gRefDef, 'y-max', undef);
	my($ymin) = graphParam($gRefDef, 'y-min', undef);
	my(@rigid);

	if (defined($ymin) || defined($ymax)) {
		push @rigid, '-r';
		push @rigid, '-u', $ymax if (defined($ymax));
		push @rigid, '-l', $ymin if (defined($ymin));
	}

	# ok, lets attempt to handle mtargets.  We need to loop through
	# each of the individual targets and construct the graph command
	# on each of those.  The other initializations should be outside
	# the loop, and the actual graph creation should be after the loop.

	my(@defs) = ();
	my(@cdefs) = ();
	my($yaxis) = "";
	my($bytes) = 0;
	my(@lines) = ();
	my($ct) = 0;
	my($usedArea) = 0;
	my(@linePushed);
	my($scaled);
		
	# prepare a dsmap, using the target and targettype dicts
	# we do this outside the loop to keep the DS map from expanding

	my($ttype) = lc($targRef->{'target-type'});
	my($ttRef) = $main::gCT->configHash($name, 'targettype', $ttype, $targRef);
	my(%dsmap) = makeDSMap($ttRef->{'ds'});

	my($path) = $targRef->{'auto-target-path'};
	my($thisName);
	foreach $thisName (@mtargets) {
		# this allows local targets to use shorter name
		$thisName = "$path/$thisName" unless ($thisName =~ /^\//);

		my($targRef) = $main::gCT->configHash($thisName, 'target', undef);
		ConfigTree::Cache::addAutoVariables($thisName, $targRef,
				$main::gConfigRoot);
		my($thisTname) = $targRef->{'auto-target-name'};

		# take the inst from the url if it's there
		my($inst) = $main::gQ->param('inst');
		if (defined($inst)) {
			$targRef->{'inst'} = $inst;
		}

		# now that inst is set right, expand it.
		ConfigTree::Cache::expandHash($targRef, $targRef);

		# Then pick up the values
		# things we pick up form the target dict
		my($rrd) = $targRef->{'rrd-datafile'};

		# use the dslist to create a set of defs/cdefs

		my($ds);
		my($dsGr);
		foreach $ds (split(/,/, $dslist)) {
			$ds = lc($ds);
			if ($ds =~s /#//) {
				$dsGr=0;
			}
			else	{
				$dsGr=1;
			}
			my($legend, $color, $colorCode, $drawAs, $scale);
	
			my($gRef) = $main::gCT->configHash($name, 'graph', $ds, $targRef);

			$legend = graphParam($gRef, 'legend', $ds);

			if (($isMTarget) && (!$isMTargetsOps)) {
				$legend .= " ($thisTname)";
			}

			$color = graphParam($gRef, 'color', nextColor($colorRef));
			$drawAs = graphParam($gRef, 'draw-as', 'LINE2');
			$drawAs = uc($drawAs);

			# only allow 1 area graph per gif
#			if ($drawAs eq "AREA")  {
#				if ($usedArea)  {
#					$drawAs = 'LINE2';
#				}  else  {
#					$usedArea = 1;
#				}
#			}
# i need more than one AREA

			$scale = graphParam($gRef, 'scale', undef);
			if (defined($scale))  {
				$scaled=1;
			}
	
			# this way, we only take the _first_ yaxis that
			# was offered to us. (If they are trying to graph
			# different things on one graph, they get what they deserve:
			# a mis-labeled graph. So there.)
			if (! $yaxis) {
				$yaxis = graphParam($gRef, 'y-axis', '');
			}

			# pick up the value for bytes, if we have not already
			# found it.
			if (! $bytes) {
				$bytes = graphParam($gRef, 'bytes', 0);
			}

			$colorCode = colorToCode($colorRef, $color);
	
			my($dsidx) = $dsmap{$ds};
			if (defined($dsidx)) {
				if ($isMTarget) {
					# use original DEF for MTarget of for scaled
					push @defs, "DEF:ds$ct=$rrd:ds$dsidx:AVERAGE";
				}
				else	{
					# use DS name
					push @defs, "DEF:$ds=$rrd:ds$dsidx:AVERAGE";
				}
				my($mod) = $ct % $numDSs;
				if (defined($scale)) {
					if ($isMTarget) {
						push @cdefs, "CDEF:sds$ct=$ds,$scale";
					}
					else	{
						push @cdefs, "CDEF:sc_$ds=$ds,$scale";
					}
					if ($isMTargetsOps) {
						if (!$linePushed[$mod])  {
							push @lines, "$drawAs:tot$mod#$colorCode:$legend";
							$linePushed[$mod] = 1;
						}
					}  else  {
						if ($dsGr) {
							push @lines, "$drawAs:sds$ct#$colorCode:$legend";
						}
					}
				} else {
					if ($isMTargetsOps)  {
						if (!$linePushed[$mod])  {
							push @lines, "$drawAs:tot$mod#$colorCode:$legend";
							$linePushed[$mod] = 1;
						}
					}  else  {
						if ($dsGr) {
							push @lines, "$drawAs:$ds#$colorCode:$legend";
						}
					}
				}
				$ct++;
			} else {
				# ERR: Unknown ds-name in dslist.
				# or special scaled graph :-)
				#$scale=lc($scale);
				push @cdefs, "CDEF:$ds=$scale";
				if ($dsGr) {
					push @lines, "$drawAs:$ds#$colorCode:$legend";	
				}
			}
	
			# tell nextColor what color we just used
			usedColor($color);
		}

	# This is the end of the loop we do for each target
	}

	# This is where we will deal with arithematic operations

	if ($isMTargetsOps)  {

		# first build the cdefs
		
		my($nameme);
		if ($scaled)  {
			$nameme = "sds";
		}  else  {
			$nameme = "ds";
		}
		my($i) = -1;
		my(@dsnames);
		while ($i < ($ct-1))  {
			$i++;
			push @{$dsnames[$i % $numDSs]}, "$nameme$i";
		}
		
		my($j) = 0;
		while ($j < $numDSs)  {
			my(@d) = @{$dsnames[$j]};
			my($str2) = "CDEF:tot$j=" .
					join(',', @d, convertOps($MTargetsOps, $#d+1));
			push @cdefs, $str2;

			$j++;
		}

		# we built the line commands earlier
	}


	# add a vrule for each "zero" time:
	#	for a daily graph, zero times are midnights
	#	for a weekly graph, zero times are Monday Midnights
	#	for a monthly graph, zero times are 1st of the month midnight
	#	for a yearly graph, zero times are 1st of the year

	my($vruleColor) = graphParam($gRefDef, 'vrule-color', undef);
	my(@vrules);
	if (defined($vruleColor) and $vruleColor ne 'none') {
		$vruleColor = colorToCode($colorRef, $vruleColor);
		
		my($rangeType) = rangeType($range);
		
		# first, find the time of the most recent zero mark
		my($timeToZeroTime) = 0;
		my($deltaZeroTime) = 0;		# the number of seconds between zero times
		my($now) = time();
		my($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($now);

		# find out how many seconds we are past the last zero time
		$timeToZeroTime += $sec;
		$timeToZeroTime += $min * 60;
		$timeToZeroTime += $hour * 60 * 60;
		$deltaZeroTime = 60 * 60 * 24;

		if ($rangeType == $main::kTypeWeekly) {
			my($numDaysToMonday) = ($wday - 1);
			$timeToZeroTime += $numDaysToMonday * 60 * 60 * 24;
			$deltaZeroTime *= 7;
		}
		if ($rangeType == $main::kTypeMonthly) {
			$timeToZeroTime += ($mday - 1) * 60 * 60 * 24;
			$deltaZeroTime *= 30;
		}
		if ($rangeType == $main::kTypeYearly) {
			$timeToZeroTime += $yday * 60 * 60 * 24;
			# yikes... what about leap year? Ick.
			$deltaZeroTime *= 365;
		}
		my($zeroTime) = $now - $timeToZeroTime;
		
		# loop and add a vrule for every zero point from
		# now back to ($now - $range). This loop has
		# the nice property that it will skip even the first VRULE,
		# if that wouldn't fit on the graph.
		while ($zeroTime > ($now - $range)) {
			push @vrules, "VRULE:$zeroTime#$vruleColor:";
			$zeroTime -= $deltaZeroTime;
		}
	}

	if ($#defs+1 == 0 || $#lines+1 == 0) {
		Error("No graph to make?");
	}

	if (! -d $main::gCacheDir) {
		mkdir($main::gCacheDir, 0777);
		chmod(0777, $main::gCacheDir);
	}

	# this sets -b based on the value of the bytes parameter
	my(@base) = ( '--base', '1000' );
	if ($bytes) {
		@base = ( '--base', '1024' );
	}

	# handle passthrough arguments
	my($pass) = graphParam($gRefDef, 'rrd-graph-args', undef);
	my(@pass) = ();
	if (defined($pass)) {
		@pass = split(/\s+/, $pass);
	}

	my(@rules, $e);
	if ($targRef->{'events'}) {
		foreach $e (split(/\s*,\s*/, $targRef->{'events'})) {
			my($evRef) = $main::gCT->configHash($name, 'event', $e, $targRef);
			if ($evRef && $evRef->{'time'}) {
				push @rules, join('', 'VRULE', ':', $evRef->{'time'},
					'#', colorToCode($colorRef, $evRef->{'color'}),
					':', $evRef->{'name'});
			}
		}
		push @rules, 'COMMENT:\s';
	}

	my(@args) = ($imageName, @rigid, @interlaced,
				 @base, @pass, @rules,
				 '--start', "-$range",
				 '--vertical-label', $yaxis,
				 '--width',          $width,
				 '--height',         $height,
				 @defs, @cdefs, @lines, @vrules);

	# we unlink the image so that if there's a failure, we
	# won't accidentally display an old image
	Debug("RRDs::graph " . join(" ", @args));
	unlink($imageName);
    my($avg, $w, $h) = RRDs::graph(@args);
	
	if (my $error = RRDs::error) {
		Warn("Unable to create graph: $error\n");
	}

    my($wh) = graphParam($gRefDef, 'width-hint', undef);
    my($hh) = graphParam($gRefDef, 'height-hint', undef);

	Warn("Actual graph width ($w) differs from width-hint ($wh).")
		if ($w && $wh && ($wh != $w));
	Warn("Actual graph height ($h) differs from height-hint ($hh).")
		if ($h && $hh && ($hh != $h));


	sprayGif($imageName);
	unlink($imageName) if $needUnlink;
}

sub tryGif {
	my($gif) = @_;
	
	# we need to make certain there are no buffering problems here.
	local($|) = 1;
	
	if (! open(GIF, "<$gif")) { 
		return;
	} else { 
		my($stuff, $len); 
		binmode(GIF);
		while ($len = read(GIF, $stuff, 8192)) { 
			print $stuff; 
		} 
		close(GIF); 
	}
	return 1;
}

sub sprayGif {
	my($gif) = @_;

	print $main::gQ->header('image/gif');

	if (! tryGif($gif)) {
		Warn("Could not open $gif: $!");
		if (! tryGif("images/failed.gif")) {
			Warn("Could not send failure gif: $!");
			return;
		}
	}

	return 1;
}

sub getHTMLDict {
	my($name, $targRef) = @_;

	my($h) = $main::gCT->configHash($name, 'html');

	$h->{'auto-long-version'} = $main::gVersion;
	my($sv) = ($main::gVersion =~ /Cricket version ([^\s]+) /);
	$sv = "?" unless ($sv);
	$h->{'auto-short-version'} = $sv;

	$h->{'auto-error'} = $main::gError;
	$h->{'auto-title'} = '';
	
	# put the contents of the target dict into the HTML dict
	# so that it will be available for expansion
	my($tag);
	foreach $tag (keys(%{$targRef})) {
		$h->{$tag} = $targRef->{$tag};
	}

	return $h;
}

sub htmlHeader {
	my($name, $targRef, $title) = @_;
	my(@headerArgs);

	my($h) = getHTMLDict($name, $targRef);
	$h->{'auto-title'} = $title;
	ConfigTree::Cache::expandHash($h, $h);
	
	print $main::gQ->header('text/html');
	print "<html>\n";
	print "<head>\n";
	if ($h->{'head'}) {
		print $h->{'head'}, "\n";
	}
	print "<meta name=\"generator\" content=\"",
		$h->{'auto-long-version'}, "\">\n";
      print "<style>\n";
	print "{ font-family: Arial }\n";
	print "</style>\n";
	print "</head>\n";
	
	my($body) = $h->{'body-options'};
	$body = "" unless (defined($body));
	
	print "<body $body>\n";
	
	if ($h->{'page-header'}) {
		print $h->{'page-header'}, "\n";
	}

	return;
}

sub htmlFooter {
	my($name, $targRef) = @_;

	my($h) = getHTMLDict($name, $targRef);
	ConfigTree::Cache::expandHash($h, $h);
	
	if ($h->{'page-footer'}) {
		print $h->{'page-footer'}, "\n";
	}

	print "</body>\n";
	print "</html>\n";
	return;
}

# routines to manage the colors

sub usedColor {
	my($c) = @_;
	my($i, @res);
	foreach $i (@main::gColors) {
		push @res, $i unless (lc($i) eq lc($c));
	}
	@main::gColors = @res;
}

sub nextColor {
	my($colorRef) = @_;

	# make the color list, when necessary	
	if (! defined(@main::gColors)) {
		if (defined($colorRef)) {
			my($order) = $colorRef->{'--order--'};
			if (! defined($order)) {
				@main::gColors = sort keys %{$colorRef};
			} else {
				@main::gColors = split(/,\s+/, $order);
			}
		} else {
			# there are no colors available...
			@main::gColors = ();
		}
	}

	my($color) = '00cc00';		# default to green if none left (or given)
	if ($#main::gColors+1 > 0) {
		$color = $main::gColors[0];
	}
	return $color;
}

sub colorToCode {
	my($colorRef, $color) = @_;
	my($code) = $colorRef->{$color};
	# if we didn't find one, then use the passed in color, assuming it's
	# a color code...
	$code = $color if (! defined($code));
	return $code;
}

# This routine chooses the right value for a graph parameter;
# If uses the default passed in, then the value from the --default--
# dict, then the value from the dict named after the ds (if given).

sub graphParam {
    my($gRef, $param, $default) = @_;

    $param = lc($param);
    my($res) = $default;

    if (defined($gRef->{$param})) {
        $res = $gRef->{$param};
    }
    return $res;
}

# make the range-size navigation links
sub makeNavLinks {
	my($r, @links);
	my(@r) = ('d', 'w', 'd:w', 'm:y', 'd:w:m:y');
	my(@rName) = ('Hourly', 'Daily', 'Short-Term', 'Long-Term', 'All');
	my($i) = 0;
	foreach $r (@r) {
		$main::gQ->param('ranges', $r[$i]);
		my($me) = $main::gQ->self_url();
		push @links, "<a href=\"$me\">$rName[$i]</a>" .
			"&nbsp;&nbsp;&nbsp;";
		$i++;
	}
	return @links;
}

sub generateImageName {
	my($q) = @_;
	my($param, $md5);

	$md5 = new MD5;

	foreach $param ($q->param()) {
		next if ($param eq 'rand');
		if ($param eq 'cache') {
			if (lc($q->param($param)) eq 'no') {
				return;
			}
		}
		$md5->add($param, $q->param($param));
	}
	my($hash) = unpack("H8", $md5->digest());

	return "$main::gCacheDir/cricket-$hash.gif";
}

sub byFirstVal {
	$a->[0] <=> $b->[0];
}

# fixHome:
# This subroutine is a bit of a hack to properly set $HOME based
# on the assumption that this script will be called via a URL that
# looks like: http://www/~cricket/grapher.cgi. If this doesn't apply
# to your installation, then you might want to simply uncomment the
# brute force method, below.

sub fixHome {

	# brute force:
	$ENV{'HOME'} = 'd:\app32bit\cricket';
	return;
	
}

sub isnan {
	return ($_[0] =~ /^NaN$/);
}

sub convertOps {
	my($mto, $num) = @_;
	if (lc($mto) eq 'sum()') {
		return join(',', '+' x ($num-1));
	}
	return $mto;
}

sub makeInstMap {
	my($targRef) = @_;

	my($ins) = $targRef->{'inst-names'};
	if (! $ins) {
		return;
	}
	my(@ins) = eval($ins);

	my($inst) = $targRef->{'inst'};

	if (! $inst) {
		$inst = '()';
	}

	my($hash) = {};
	my($ct) = 0;
	my($i);

	my(@inst) = eval($inst);
	if ($#inst+1 > 0) {
		# they gave us an inst array, so match them up one to
		# one.
		foreach $i (@inst) {
			$hash->{$i} = $ins[$ct++];
		}
		Debug("inst array is: ", join(", ", @inst));
	} else {
		# there's 0 or 1 inst's, so make a simple table
		foreach $i (@ins) {
			$hash->{$ct++} = $i;
		}
	}

	return $hash;
}
